#include "game.h"

//Used to blit images to the screen
void blitImage(int x, int y, SDL_Surface* source, SDL_Surface* destination, SDL_Rect* clip){
    SDL_Rect offset;
    offset.x = x;
    offset.y = y;
    SDL_BlitSurface(source, clip, destination, &offset);
}

//Used to load images and remove all white
SDL_Surface *loadImage(string file){
	SDL_Surface *load = NULL;
	SDL_Surface *optimized = NULL;
	load = IMG_Load(file.c_str());
	optimized = SDL_DisplayFormat(load);
	SDL_FreeSurface(load);
	Uint32 colorkey = SDL_MapRGB(optimized->format, 0xFF, 0x00, 0xFF);
	SDL_SetColorKey(optimized, SDL_SRCCOLORKEY, colorkey);
	return optimized;
}

void flip(int &x){
	if(x == 1)
		x = 0;
	else
		x = 1;
}

Map::Map(string file){
	terrain = loadImage("images/terrain.png");
	lvlitems = loadImage("images/lvlitems.png");
	items = loadImage("images/items.png");
	other = loadImage("images/other.png");
	background = loadImage("images/background.png");
	hud = loadImage("images/hud.png");
	for(int i = 0; i < NUMLIGHT; ++i)
		for(int j = 0; j < NUMTILES; ++j){
			sprite[i][j].x = j * TILESIZE;
			sprite[i][j].y = i * TILESIZE;
			sprite[i][j].w = sprite[i][j].h = TILESIZE;
		}
	ifstream in(file.c_str());
	anim = frame = 0;
	int item;
	char junk;
	int pow;
	in >> mapSize.x >> junk >> mapSize.y;
	in >> pLoc.x >> junk >> pLoc.y;
	player.at.x = pLoc.x * TILESIZE;
	player.at.y = pLoc.y * TILESIZE;
	for(int i = 0; i < mapSize.x; ++i)
		for(int j = 0; j < mapSize.y; ++j){
			tile[i][j].loc.x = i;
			tile[i][j].loc.y = j;
			tile[i][j].light = tile[i][j].laserH = tile[i][j].laserV = 0;
			tile[i][j].power = false;
			tile[i][j].chest = -1;
			for(int k = 0; k < LVLITEMS; ++k)
				tile[i][j].item[k] = false;
			tile[i][j].traversable = true;
			tile[i][j].unlocked = false;
			in >> tile[i][j].tile >> junk;
			if(tile[i][j].tile == WALL || tile[i][j].tile == PWALL)
				tile[i][j].traversable = false;
			if(tile[i][j].tile == SPACE || tile[i][j].tile == PSPACE){
				in >> item >>junk;
				if(item != -1){
					tile[i][j].item[item] = true;
					tile[i][j].traversable = false;
					if(item == CHEST){
						in >> item >> junk;
						tile[i][j].chest = item;
					}
					else if(item == SWITCH){
						in >> pow >> junk;
						if(pow == POWERON){
							powerStack.insert(&tile[i][j]);
						}
					}
				}
			}
		}
	in.close();
	//Get the light source info from light.info
	string garbage;
	in.open("data/light.info");
	getline(in, garbage);
	getline(in, garbage);
	for(int i = 0; i < LIGHTSIZE; ++i)
		for(int j = 0; j < LIGHTSIZE; ++j)
			in >> lightsrc[i][j];
	getline(in, garbage);
	getline(in, garbage);
	for(int i = 0; i < LANTERNSIZE; ++i)
		for(int j = 0; j < LANTERNSIZE; ++j)
			in >> lantern[i][j];
	in.close();
	Tile *t;
	while(!powerStack.isEmpty()){
		t = powerStack.remove();
		if(t->power == false)
			power(t->loc.x, t->loc.y);
	}
}

Map::~Map(){
	SDL_FreeSurface(terrain);
	SDL_FreeSurface(lvlitems);
	SDL_FreeSurface(items);
	SDL_FreeSurface(other);
	SDL_FreeSurface(background);
	SDL_FreeSurface(hud);
}

void Map::blit(SDL_Surface *screen){
	//animate
	++anim;
	if(anim > 2){
		anim = 0;
		flip(frame);
	}
	//blit a black background
	for(int i = 0; i < screen->w; i += 200)
		for(int j = 0; j < screen->h; j += 200)
			blitImage(i, j, background, screen);
	//figure out where to start blitting tiles
	int maxlight;
	int x = (player.at.x - (screen->h / 2) - (TILESIZE / 2) + TILESIZE) * -1;
	int y = (player.at.y - (screen->w / 2) - (TILESIZE / 2) + TILESIZE) * -1;
	for(int i = 0; i < mapSize.x; ++i)
		for(int j = 0; j < mapSize.y; ++j){
			//Light of the tile
			maxlight = tile[i][j].light;
			if(maxlight >= NUMLIGHT)
				maxlight = NUMLIGHT - 1;
			blitImage(j * 32 + y, i * 32 + x, terrain, screen, &sprite[maxlight][tile[i][j].tile]);
			//Power and lasers
			if(tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, other, screen, &sprite[0][rand() % 2 + 1]);
			if(tile[i][j].laserH > 0)
				blitImage(j * 32 + y, i * 32 + x, other, screen, &sprite[1][0]);
			if(tile[i][j].laserV > 0)
				blitImage(j * 32 + y, i * 32 + x, other, screen, &sprite[1][1]);
			//Level items on the map
			if(tile[i][j].item[SWITCH] && tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[1][SWITCH]);
			else if(tile[i][j].item[SWITCH] && !tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[0][SWITCH]);
			else if(tile[i][j].item[CHEST] && tile[i][j].chest == -1 && tile[i][j].light != 0)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[1][CHEST]);
			else if(tile[i][j].item[CHEST] && tile[i][j].chest != -1 && tile[i][j].light != 0)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[0][CHEST]);
			else if(tile[i][j].item[DOOR] && tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[1][DOOR]);
			else if(tile[i][j].item[DOOR] && !tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[0][DOOR]);
			else if(tile[i][j].item[LOCKDOOR] && tile[i][j].unlocked)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[1][LOCKDOOR]);
			else if(tile[i][j].item[LOCKDOOR] && !tile[i][j].unlocked)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[0][LOCKDOOR]);
			else if(tile[i][j].item[LIGHTSRC] && tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[1][LIGHTSRC]);
			else if(tile[i][j].item[LIGHTSRC] && !tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[0][LIGHTSRC]);
			else if(tile[i][j].item[LASER] && tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[1][LASER]);
			else if(tile[i][j].item[LASER] && !tile[i][j].power)
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[0][LASER]);
			else if(tile[i][j].item[GEM] && tile[i][j].light != 0){
				blitImage(j * 32 + y, i * 32 + x, lvlitems, screen, &sprite[frame][GEM]);
				if(rand() % 40 < 2)
					blitImage(j * 32 + y, i * 32 + x, other, screen, &sprite[1][2]);
			}
			else if((tile[i][j].item[GEM] || tile[i][j].item[CHEST]) && tile[i][j].light == 0)
				blitImage(j * 32 + y, i * 32 + x, other, screen, &sprite[frame][3]);
		}
	blitImage((screen->w / 2) - 144, screen->h - TILESIZE, hud, screen);
	for(int i = 32, j = 0; i < 32 * 9; i += 64, ++j){
		blitImage((screen->w / 2) + i - 144, screen->h - TILESIZE, other, screen, &sprite[0][0]);
		if(player.inventory[j])
			blitImage((screen->w / 2) + i - 144, screen->h - TILESIZE, items, screen, &sprite[0][j]);
	}
	player.translate();
	player.blit(screen);
}

//blit the gem above the player and wait for a second
void Map::blitWin(SDL_Surface *screen){
	blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2), player.player, screen, &sprite[0][SPECIAL]);
	blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2) - TILESIZE, lvlitems, screen, &sprite[0][GEM]);
	SDL_Flip(screen);
	SDL_Delay(1000);
}

//blit the player dead image and wait for a second
void Map::blitDead(SDL_Surface *screen){
	blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2), player.player, screen, &sprite[1][SPECIAL]);
	SDL_Flip(screen);
	SDL_Delay(1000);
}

//blit the picked up item above the player for a second
void Map::blitItem(SDL_Surface *screen, int item, int &timer){
	blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2), player.player, screen, &sprite[0][SPECIAL]);
	blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2) - TILESIZE, items, screen, &sprite[0][item]);
	SDL_Flip(screen);
	SDL_Delay(1000);
	timer += 1000;
}

//clear everything off the map to load the next level
void Map::clearMap(){
	for(int i = 0; i < mapSize.x; ++i)
		for(int j = 0; j < mapSize.y; ++j){
			tile[i][j].tile = tile[i][j].light = 0;
			tile[i][j].laserH = tile[i][j].laserV = 0;
			tile[i][j].chest = -1;
			for(int k = 0; k < LVLITEMS; ++k)
				tile[i][j].item[k] = false;
			tile[i][j].traversable = true;
			tile[i][j].power = false;
			tile[i][j].unlocked = false;
		}
}

bool Map::handleEvent(SDL_Event &event, bool &win, bool &dead, SDL_Surface *screen, int &timer){
	if(event.type == SDL_KEYDOWN && !player.moving)
		switch(event.key.keysym.sym){
			case SDLK_UP:
				player.direction = UP;
				if(canMove(UP)){
					checkTile(dead);
					lamp(false);
					player.move(UP);
					pLoc.x -= 1;
					lamp(true);
					player.held = true;
				}
				break;
			case SDLK_DOWN:
				player.direction = DOWN;
				if(canMove(DOWN)){
					checkTile(dead);
					lamp(false);
					player.move(DOWN);
					pLoc.x += 1;
					lamp(true);
					player.held = true;
				}
				break;
			case SDLK_LEFT:
				player.direction = LEFT;
				if(canMove(LEFT)){
					checkTile(dead);
					lamp(false);
					player.move(LEFT);
					pLoc.y -= 1;
					lamp(true);
					player.held = true;
				}
				break;
			case SDLK_RIGHT:
				player.direction = RIGHT;
				if(canMove(RIGHT)){
					checkTile(dead);
					lamp(false);
					player.move(RIGHT);
					pLoc.y += 1;
					lamp(true);
					player.held = true;
				}
				break;
			case SDLK_SPACE:
				action(win, screen, timer);
				break;
			case SDLK_ESCAPE: return true; break;
			case SDLK_k: dead = true; break;
		}
	else if(event.type == SDL_KEYUP)
		switch(event.key.keysym.sym){
			case SDLK_UP: player.held = false; break;
			case SDLK_DOWN: player.held = false; break;
			case SDLK_LEFT: player.held = false; break;
			case SDLK_RIGHT: player.held = false; break;
		}
	return false;
}

//continuous movement if button held
void Map::heldMovement(bool &win, bool &dead, bool &quit){
	if(player.held && !player.moving)
		if(canMove(player.direction)){
			checkTile(dead);
			lamp(false);
			player.move(player.direction);
			if(player.direction == UP)
				pLoc.x -= 1;
			else if(player.direction == DOWN)
				pLoc.x += 1;
			else if(player.direction == LEFT)
				pLoc.y -= 1;
			else //RIGHT
				pLoc.y += 1;
			lamp(true);
		}
}

//Checks if the tile is a trap (create wall) or a laser (player dies)
void Map::checkTile(bool &dead){
	if(tile[pLoc.x][pLoc.y].tile == TRAPWALL && !player.inventory[BOOTS]){
		tile[pLoc.x][pLoc.y].tile = WALL;
		tile[pLoc.x][pLoc.y].traversable = false;
		tile[pLoc.x][pLoc.y].laserV = tile[pLoc.x][pLoc.y].laserH = 0;
	}
	if(tile[pLoc.x][pLoc.y].laserH > 0 || tile[pLoc.x][pLoc.y].laserV > 0)
		if(!player.inventory[INVISPOT])
			dead = true;
}

//Checks if the player has walked on a laser (This function kills instantly)
void Map::checkDead(bool &dead){
	if(!player.moving)
		if(tile[pLoc.x][pLoc.y].laserH > 0 || tile[pLoc.x][pLoc.y].laserV > 0)
			if(!player.inventory[INVISPOT])
				dead = true;
}

bool Map::canMove(int dir){
	int x = pLoc.x;
	int y = pLoc.y;
	if(dir == UP)
		x -= 1;
	else if(dir == DOWN)
		x += 1;
	else if(dir == LEFT)
		y -= 1;
	else
		y += 1;
	if(inbounds(x, y))
		return tile[x][y].traversable;
	return false;
}

void Map::action(bool &win, SDL_Surface *screen, int &timer){
	//check tile in front of player
	Location check = pLoc;
	if(player.direction == UP)
		check.x -= 1;
	else if(player.direction == DOWN)
		check.x += 1;
	else if(player.direction == LEFT)
		check.y -= 1;
	else //RIGHT
		check.y += 1;
	if(inbounds(check.x, check.y)){
		//if chest add item to player
		if(tile[check.x][check.y].item[CHEST] && tile[check.x][check.y].light != 0 && tile[check.x][check.y].chest != -1){
			player.inventory[tile[check.x][check.y].chest] = true;
			if(player.inventory[INVISPOT])
				player.invisible = true;
			else if(player.inventory[LANTERN] && tile[check.x][check.y].chest == LANTERN)
				lamp(true);
			blitItem(screen, tile[check.x][check.y].chest, timer);
			tile[check.x][check.y].chest = -1;
		}
		//if switch flip power
		else if(tile[check.x][check.y].item[SWITCH]){
			power(check.x, check.y);
		}
		//if gem win
		else if(tile[check.x][check.y].item[GEM] && tile[check.x][check.y].light != 0){
			tile[check.x][check.y].item[GEM] = false;
			win = true;
		}
		//if locked door and have key open
		else if(tile[check.x][check.y].item[LOCKDOOR]){
			if(player.inventory[KEY]){
				tile[check.x][check.y].unlocked = true;
				tile[check.x][check.y].traversable = true;
			}
		}
	}
}

//Check if a tile is inbounds (prevents segfaults)
bool Map::inbounds(int x, int y){
	if(x >= 0 && x < mapSize.x && y >= 0 && y < mapSize.y)
		return true;
	return false;
}

//Turns the lamp on or off around the player
void Map::lamp(bool light){
	if(player.inventory[LANTERN])
		for(int i = pLoc.x - 2, I = 0; i <= pLoc.x + 2; ++i, ++I)
			for(int j = pLoc.y - 2, J = 0; j <= pLoc.y + 2; ++j, ++J)
				if(inbounds(i, j)){
					if(light)
						tile[i][j].light += lantern[I][J];
					else
						tile[i][j].light -= lantern[I][J];
				}
}

//Recursively turns the power on to all connected blocks
void Map::power(int x, int y){
	tile[x][y].power = !tile[x][y].power;
	if(tile[x][y].item[LIGHTSRC])
		light(x, y);
	else if(tile[x][y].item[LASER])
		laser(x, y);
	else if(tile[x][y].item[DOOR])
		door(x, y);
	if(inbounds(x + 1, y))
		if(tile[x + 1][y].tile == PWALL || tile[x + 1][y].tile == PSPACE)
			if(tile[x + 1][y].power != tile[x][y].power)
				power(x + 1, y);
	if(inbounds(x - 1, y))
		if(tile[x - 1][y].tile == PWALL || tile[x - 1][y].tile == PSPACE)
			if(tile[x - 1][y].power != tile[x][y].power)
				power(x - 1, y);
	if(inbounds(x, y + 1))
		if(tile[x][y + 1].tile == PWALL || tile[x][y + 1].tile == PSPACE)
			if(tile[x][y + 1].power != tile[x][y].power)
				power(x, y + 1);
	if(inbounds(x, y - 1))
		if(tile[x][y - 1].tile == PWALL || tile[x][y - 1].tile == PSPACE)
			if(tile[x][y - 1].power != tile[x][y].power)
				power(x, y - 1);
}

//Turn on or off a light based on the power of the tile
void Map::light(int x, int y){
	for(int i = x - 3, I = 0; i <= x + 3; ++i, ++I)
		for(int j = y - 3, J = 0; j <= y + 3; ++j, ++J)
			if(inbounds(i, j)){
				if(tile[x][y].power)
					tile[i][j].light += lightsrc[I][J];
				else
					tile[i][j].light -= lightsrc[I][J];
			}
}

void Map::laser(int x, int y){
	int laser = 1;
	if(!tile[x][y].power)
		laser = -1;
	int X = x + 1, Y = y;
	while(inbounds(X, Y) && (tile[X][Y].traversable || tile[X][Y].item[LOCKDOOR] || tile[X][Y].item[DOOR])){
		tile[X][Y].laserV += laser;
		++X;
	}
	X = x - 1, Y = y;
	while(inbounds(X, Y) && (tile[X][Y].traversable || tile[X][Y].item[LOCKDOOR] || tile[X][Y].item[DOOR])){
		tile[X][Y].laserV += laser;
		--X;
	}
	X = x, Y = y + 1;
	while(inbounds(X, Y) && (tile[X][Y].traversable || tile[X][Y].item[LOCKDOOR] || tile[X][Y].item[DOOR])){
		tile[X][Y].laserH += laser;
		++Y;
	}
	X = x, Y = y - 1;
	while(inbounds(X, Y) && (tile[X][Y].traversable || tile[X][Y].item[LOCKDOOR] || tile[X][Y].item[DOOR])){
		tile[X][Y].laserH += laser;
		--Y;
	}
}

void Map::door(int x, int y){
	if(tile[x][y].power)
		tile[x][y].traversable = true;
	else
		tile[x][y].traversable = false;
}

Player::Player(){
	player = loadImage("images/player.png");
	for(int i = 0; i < CHARHEIGHT; ++i)
		for(int j = 0; j < CHARWIDTH; ++j){
			sprite[i][j].x = j * TILESIZE;
			sprite[i][j].y = i * TILESIZE;
			sprite[i][j].w = sprite[i][j].h = TILESIZE;
		}
	direction = DOWN;
	frame = 0;
	anim = 0;
	speed = SPEED;
	moving = invisible = false;
	for(int i = 0; i < INVITEMS; ++i)
		inventory[i] = false;
	held = false;
}

Player::~Player(){
	SDL_FreeSurface(player);
}

void Player::blit(SDL_Surface *screen){
	if(invisible)
		blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2), player, screen, &sprite[frame][INVIS]);
	else
		blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2), player, screen, &sprite[frame][direction]);
}

void Player::move(int dir){
	to = at;
	moving = true;
	if(dir == UP)
		to.x -= 32;
	else if(dir == DOWN)
		to.x += 32;
	else if(dir == LEFT)
		to.y -= 32;
	else
		to.y += 32;
}

void Player::translate(){
	if(moving){
		++anim;
		if(anim > 1){
			anim = 0;
			flip(frame);
		}
		if(direction == UP){
			at.x -= speed;
			if(at.x < to.x){
				at.x = to.x;
				moving = false;
			}
		}
		else if(direction == DOWN){
			at.x += speed;
			if(at.x > to.x){
				at.x = to.x;
				moving = false;
			}
		}
		else if(direction == LEFT){
			at.y -= speed;
			if(at.y < to.y){
				at.y = to.y;
				moving = false;
			}
		}
		else{
			at.y += speed;
			if(at.y > to.y){
				at.y = to.y;
				moving = false;
			}
		}
		if(!moving)
			frame = 0;
	}
}

Stack::Stack(){
	top = NULL;
}

//doesn't delete tiles because map handles that
Stack::~Stack(){
	while(!isEmpty())
		remove();
}

bool Stack::isEmpty(){
	return !top;
}

//points at tiles in map not new tiles
void Stack::insert(Tile *t){
	Node *n = new Node;
	n->t = t;
	n->next = top;
	top = n;
}

Tile *Stack::remove(){
	Tile *t = top->t;
	top->t = NULL;
	Node *victim = top;
	top = top->next;
	delete victim;
	return t;
}
