#include "editor.h"

//Converts an int to a string for use with file output
string iToS(int x){
	stringstream str;
	str << x;
	return str.str();
}

Editor::Editor(int level){
	//Set up sprite sheet
	for(int i = 0; i < NUMLIGHT; ++i)
		for(int j = 0; j < NUMTILES; ++j){
			sprite[i][j].x = j * TILESIZE;
			sprite[i][j].y = i * TILESIZE;
			sprite[i][j].w = sprite[i][j].h = TILESIZE;
		}
	sLoc.x = sLoc.y = pStart.x = pStart.y = 0;
	mapSize.x = mapSize.y = MAXMAPSIZE - 1;
	//Load images
	terrain = loadImage("images/terrain.png");
	lvlitems = loadImage("images/lvlitems.png");
	items = loadImage("images/items.png");
	other = loadImage("images/other.png");
	background = loadImage("images/background.png");
	//Opens the next level in custom levels
	file = "customlevels/custom";
	file += iToS(level);
	file += ".lvl";
	//Set up a blank map
	for(int i = 0; i < MAXMAPSIZE; ++i)
		for(int j = 0; j < MAXMAPSIZE; ++j){
			tile[i][j].tile = SPACE;
			tile[i][j].chest = -1;
			for(int k = 0; k < LVLITEMS; ++k)
				tile[i][j].item[k] = false;
			tile[i][j].power = false;
			tile[i][j].laserH = tile[i][j].laserV = 0;
		}
}

Editor::~Editor(){
	SDL_FreeSurface(terrain);
	SDL_FreeSurface(lvlitems);
	SDL_FreeSurface(items);
	SDL_FreeSurface(other);
	SDL_FreeSurface(background);
}

//opens a level for editing
void Editor::edit(){
	int item, pow;
	char junk;
	in.open(file.c_str());
	in >> mapSize.x >> junk >> mapSize.y;
	in >> pStart.x >> junk >> pStart.y;
	for(int i = 0; i < mapSize.x; ++i)
		for(int j = 0; j < mapSize.y; ++j){
			tile[i][j].loc.x = i;
			tile[i][j].loc.y = j;
			tile[i][j].light = tile[i][j].laserH = tile[i][j].laserV = 0;
			tile[i][j].power = false;
			tile[i][j].chest = -1;
			for(int k = 0; k < LVLITEMS; ++k)
				tile[i][j].item[k] = false;
			tile[i][j].traversable = true;
			tile[i][j].unlocked = false;
			in >> tile[i][j].tile >> junk;
			if(tile[i][j].tile == WALL || tile[i][j].tile == PWALL)
				tile[i][j].traversable = false;
			if(tile[i][j].tile == SPACE || tile[i][j].tile == PSPACE){
				in >> item >> junk;
				if(item != -1){
					tile[i][j].item[item] = true;
					tile[i][j].traversable = false;
					if(item == CHEST){
						in >> item >> junk;
						tile[i][j].chest = item;
					}
					else if(item == SWITCH){
						in >> pow >> junk;
						if(pow == POWERON){
							tile[i][j].power = true;
						}
					}
				}
			}
		}
	in.close();
	--mapSize.x;
	--mapSize.y;
	if(inbounds(mapSize.x + 1, 0))
		for(int i = 0; i < mapSize.y + 2; ++i)
				tile[mapSize.x + 1][i].laserH = 1;
	if(inbounds(0, mapSize.y + 1))
		for(int i = 0; i < mapSize.x + 2; ++i)
			tile[i][mapSize.y + 1].laserV = 1;
}

//Blits the entire map to the screen
void Editor::blit(SDL_Surface *screen){
	//blit a black background
	for(int i = 0; i < screen->w; i += 200)
		for(int j = 0; j < screen->h; j += 200)
			blitImage(i, j, background, screen);
	for(int i = 0, I = (screen->h / 2) - (TILESIZE / 2) - (sLoc.x * TILESIZE); i < MAXMAPSIZE; ++i, I += TILESIZE)
		for(int j = 0, J = (screen->w / 2) - (TILESIZE / 2) - (sLoc.y * TILESIZE); j < MAXMAPSIZE; ++j, J += TILESIZE)
			if(inbounds(i, j)){
				blitImage(J, I, terrain, screen, &sprite[0][tile[i][j].tile]);
				for(int k = 0; k < LVLITEMS; ++k){
					if(tile[i][j].item[k])
						blitImage(J, I, lvlitems, screen, &sprite[0][k]);
					if(tile[i][j].item[CHEST] && tile[i][j].chest == -1)
						blitImage(J, I, lvlitems, screen, &sprite[1][CHEST]);
					else if(tile[i][j].item[CHEST])
						blitImage(J, I, items, screen, &sprite[0][tile[i][j].chest]);
					if(tile[i][j].power)
						blitImage(J, I, other, screen, &sprite[0][2]);
					if(tile[i][j].laserH > 0)
						blitImage(J, I, other, screen, &sprite[1][0]);
					if(tile[i][j].laserV > 0)
						blitImage(J, I, other, screen, &sprite[1][1]);
				}
				if(i == pStart.x && j == pStart.y)
					blitImage(J, I, items, screen, &sprite[0][4]);
			}
	blitImage((screen->w / 2) - (TILESIZE / 2), (screen->h / 2) - (TILESIZE / 2), other, screen, &sprite[0][0]);
}

//Check if a tile is inbounds (prevents segfaults)
bool Editor::inbounds(int x, int y){
	if(x >= 0 && x < MAXMAPSIZE && y >= 0 && y < MAXMAPSIZE)
		return true;
	return false;
}

//Clears the currently selected tile
void Editor::clearTile(){
	tile[sLoc.x][sLoc.y].tile = 0;
	tile[sLoc.x][sLoc.y].power = false;
	tile[sLoc.x][sLoc.y].chest = -1;
	clearItem();
}

//Clears the entire map
void Editor::clear(){
	for(int i = 0; i < MAXMAPSIZE; ++i)
		for(int j = 0; j < MAXMAPSIZE; ++j){
			tile[i][j].tile = SPACE;
			tile[i][j].chest = -1;
			for(int k = 0; k < LVLITEMS; ++k)
				tile[i][j].item[k] = false;
			tile[i][j].power = false;
			tile[i][j].laserH = tile[i][j].laserV = 0;
		}
	mapSize.x = mapSize.y = sLoc.x = sLoc.y = pStart.x = pStart.y = 0;
}

//Clears the items on the current tile
void Editor::clearItem(){
	tile[sLoc.x][sLoc.y].power = false;
	for(int i = 0; i < LVLITEMS; ++i)
		tile[sLoc.x][sLoc.y].item[i] = false;
}

//Checks if the tile is "check" and sets the item "set"
void Editor::setItem(int check, int set){
	if(tile[sLoc.x][sLoc.y].tile == check){
		clearItem();
		tile[sLoc.x][sLoc.y].item[set] = !tile[sLoc.x][sLoc.y].item[set];
	}
}

//Turns power on/off for switch or cycles through items if chest
void Editor::setOther(){
	if(tile[sLoc.x][sLoc.y].item[CHEST]){
		++tile[sLoc.x][sLoc.y].chest;
		if(tile[sLoc.x][sLoc.y].chest > 3)
			tile[sLoc.x][sLoc.y].chest = -1;
	}
	else if(tile[sLoc.x][sLoc.y].item[SWITCH])
		tile[sLoc.x][sLoc.y].power = !tile[sLoc.x][sLoc.y].power;
}

//Set the edge of the map
void Editor::setSize(){
	for(int i = 0; i < MAXMAPSIZE; ++i)
		for(int j = 0; j < MAXMAPSIZE; ++j)
			tile[i][j].laserV = tile[i][j].laserH = 0;
	mapSize.x = sLoc.x;
	mapSize.y = sLoc.y;
	if(inbounds(sLoc.x + 1, 0))
		for(int i = 0; i < sLoc.y + 2; ++i)
				tile[sLoc.x + 1][i].laserH = 1;
	if(inbounds(0, sLoc.y + 1))
		for(int i = 0; i < sLoc.x + 2; ++i)
			tile[i][sLoc.y + 1].laserV = 1;
}

//Handles all of the editor input
void Editor::handleEvent(SDL_Surface *screen, SDL_Event &event, bool &quit, bool &save){
	if(event.type == SDL_KEYDOWN)
		switch(event.key.keysym.sym){
			case SDLK_UP: if(inbounds(sLoc.x - 1, sLoc.y)) --sLoc.x; break;
			case SDLK_DOWN: if(inbounds(sLoc.x + 1, sLoc.y)) ++sLoc.x; break;
			case SDLK_LEFT: if(inbounds(sLoc.x, sLoc.y - 1)) --sLoc.y; break;
			case SDLK_RIGHT: if(inbounds(sLoc.x, sLoc.y + 1)) ++sLoc.y; break;
			case SDLK_ESCAPE: quit = true; break;
			case SDLK_BACKSPACE: clear(); break;
			case SDLK_p: pStart.x = sLoc.x; pStart.y = sLoc.y; break;
			case SDLK_o: setSize(); break;
			case SDLK_1: clearTile(); tile[sLoc.x][sLoc.y].tile = 0; break;
			case SDLK_2: clearTile(); tile[sLoc.x][sLoc.y].tile = 1; break;
			case SDLK_3: clearTile(); tile[sLoc.x][sLoc.y].tile = 2; break;
			case SDLK_4: clearTile(); tile[sLoc.x][sLoc.y].tile = 3; break;
			case SDLK_5: clearTile(); tile[sLoc.x][sLoc.y].tile = 4; break;
			case SDLK_a: setItem(PSPACE, SWITCH); break;
			case SDLK_q: setItem(SPACE, CHEST); break;
			case SDLK_s: setItem(PSPACE, DOOR); break;
			case SDLK_w: setItem(SPACE, LOCKDOOR); break;
			case SDLK_d: setItem(PSPACE, LIGHTSRC); break;
			case SDLK_f: setItem(PSPACE, LASER); break;
			case SDLK_e: setItem(SPACE, GEM); break;
			case SDLK_z: setOther(); break;
		}
}

//Writes the created level to a file
void Editor::outputLevel(){
	//Open a blank file for the new level
	out.open(file.c_str());
	out << mapSize.x + 1 << ',' << mapSize.y + 1 << '\n';
	out << pStart.x << ',' << pStart.y << '\n';
	for(int i = 0; i < mapSize.x + 1; ++i){
		for(int j = 0; j < mapSize.y + 1; ++j){
			out << tile[i][j].tile << '.';
			if(tile[i][j].tile == SPACE || tile[i][j].tile == PSPACE){
				bool found = false;
				for(int k = 0; k < LVLITEMS; ++k)
					if(tile[i][j].item[k]){
						found = true;
						out << k << '.';
					}
				if(!found)
					out << "-1.";
				if(tile[i][j].item[CHEST])
					out << tile[i][j].chest << '.';
				else if(tile[i][j].item[SWITCH]){
					if(tile[i][j].power)
						out << "1.";
					else
						out << "0.";
				}
			}
		}
		out << '\n';
	}
	out.close();
}
